/* 
 *         GA Net - a genetic algorithm for generating Network Intusion Rules
 *
 *       Copyright (C) 2010 Brian E. Lavender <brian@brie.com> 
 *
 *                     http://www.brie.com/brian/ganet/ 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "types.h"
#include "print.h"
#include "read_bsm.h"
#include "service_attacks.h"

#define BUF_SZ 80

#ifndef SWAP_4
#define SWAP_4(x) ( ((x) << 24) | \
         (((x) << 8) & 0x00ff0000) | \
         (((x) >> 8) & 0x0000ff00) | \
         ((x) >> 24) )
#endif

void destroyL_key(gpointer foo) {
  g_slice_free1(sizeof(gint),foo);
}

void destroyL_value(gpointer foo) {
  g_slice_free1(sizeof(gint),foo);
}

/**
 * g_char_hash:
 * @v: a pointer to a #gchar key
 *
 * Converts a pointer to a #gchar to a hash value.
 * It can be passed to g_hash_table_new() as the @hash_func parameter,
 * when using pointers to integers values as keys in a #GHashTable.
 *
 * Returns: a hash value corresponding to the key.
 */
guint
g_char_hash (gconstpointer v)
{
  // Copy the value of unsigned char to unsigned int
  guchar a;
  guint b;
  a = *(const guchar*) v;
  b = a;
  return b;
}

gboolean
g_char_equal (gconstpointer v1,
	      gconstpointer v2)
{
  return *((const guchar*) v1) == *((const guchar*) v2) ;
}

void 
destroyC_key(gpointer foo) {
  g_slice_free1(sizeof(gchar),foo);
}


void updateL( GHashTable *uniqueL, GHashTable *uniqueC[], guint value ) {
  guint *c, *d; // Pointers to key and value for insertion
  gchar *cc;
  guint newcnt; // new count
  guint *lkey, *lvalue; // lookup key and lookup value
  //  gchar *lckey;
  guint initcnt = 1; // initial count
  IPAddr myUnion;
  int i;

  myUnion.full = value;

  c = (guint *)g_slice_alloc0(  sizeof(guint) );
  d = (guint *)g_slice_alloc0(  sizeof(guint) );
  g_memmove(c, &value, sizeof(gint));

  if ( g_hash_table_lookup_extended( uniqueL, c, (gpointer)&lkey, 
				     (gpointer)&lvalue) ) {
    newcnt = *lvalue + 1;
    g_memmove(d, &newcnt,sizeof(guint));
  } else {
    g_memmove(d, &initcnt,sizeof(guint));
  }
  // FIXME: check to see that insert succeeded. If not, free c and d.
  g_hash_table_insert(uniqueL, c, d);

  for (i=0; i<SUBH; i++) {
    cc = (gchar *)g_slice_alloc0(  sizeof(gchar) );
    d = (guint *)g_slice_alloc0(  sizeof(guint) );
    g_memmove(cc, &(myUnion.octet[i]),sizeof(gchar)); 

    if ( g_hash_table_lookup_extended( uniqueC[i], cc, (gpointer)&lkey, 
				       (gpointer)&lvalue) ) {
      newcnt = *lvalue + 1;
      g_memmove(d, &newcnt,sizeof(guint));
    } else {
      g_memmove(d, &initcnt,sizeof(guint));
    }
    g_hash_table_insert(uniqueC[i], cc, d);
  }
  
}

// foo - raw line for input string
// length - length of input string
// myind - pointer to individual to be created.
void make_toks(char *foo, int length, individual *myind, GHashTable *uniqueL[NUM_HTABLES], GHashTable *uniqueC[NUM_HTABLES][SUBH])
{
  time_stamp ts;
  IPAddr ip;
  char *svptr1, *svptr2;
  char delims[] = " ";
  char durdelim[] = ":";
  char ipdelim[] = ".";
  char *result = NULL;
  char *result2 = NULL;
  int idx = 0;
  int idx2 = 0;
  int i;
  int matched;
  int pv; // parsed value 
  char tmp[BUF_SZ];
  gboolean getUnique;

  init_attacks();

  if (uniqueL != NULL && uniqueC != NULL )
    getUnique = TRUE;
  else 
    getUnique = FALSE;
  
  foo[length - 1] = '\0';
 
  result = strtok_r( foo, delims, &svptr1 );

  while( result != NULL ) {
    switch (idx) {

    case F_DURATION: // duration
      ts.byte[0] = -1;
      idx2 = 1;
      strncpy(tmp, result, BUF_SZ);
      result2 = strtok_r( tmp , durdelim, &svptr2 );
      while( result2 != NULL ) {
	ts.byte[idx2] = atoi(result2);
	result2 = strtok_r(NULL, durdelim, &svptr2 );
	idx2++;
      }
      myind->chrome[G_DURATION] = ts.tot;
      if (getUnique) 
	updateL(uniqueL[G_DURATION], uniqueC[G_DURATION], myind->chrome[G_DURATION]);
      break; // end of duration

    case F_SERVICE: // service 
      for (i = 0; i < ENDP; i++) {
	if ( strncmp(services[i], result, strlen( services[i] ) ) == 0 ) {
	  myind->chrome[G_SERVICE] = i;
	  if (getUnique) 
	    updateL(uniqueL[G_SERVICE], uniqueC[G_SERVICE], myind->chrome[G_SERVICE]);
	}
      }
      break;// end of service 

    case F_SOURCE_PORT: // source port
      pv = atoi(result);
      myind->chrome[G_SOURCE_PORT] = pv; 
      if (getUnique) 
	updateL(uniqueL[G_SOURCE_PORT], uniqueC[G_SOURCE_PORT], myind->chrome[G_SOURCE_PORT]);
      break; // end of source port

    case F_DEST_PORT: // destination port
      myind->chrome[G_DEST_PORT] = atoi(result);
      if (getUnique) 
	updateL(uniqueL[G_DEST_PORT], uniqueC[G_DEST_PORT], myind->chrome[G_DEST_PORT]);  
      break; // end of destination port

    case F_SRC_IP: // Source IP
      idx2 = 0; 
      strncpy(tmp, result, BUF_SZ); // copy up to the size of the target buffer
      result2 = strtok_r( tmp , ipdelim, &svptr2 );
      while( result2 != NULL ) {	
	ip.octet[idx2] = atoi(result2);      
	result2 = strtok_r(NULL, ipdelim, &svptr2 );
	idx2++;
      }
      myind->chrome[G_SRC_IP] = ip.full;
      if (getUnique) 
	updateL(uniqueL[G_SRC_IP], uniqueC[G_SRC_IP], myind->chrome[G_SRC_IP]);
      break; // end of Source IP

    case F_DEST_IP: // Dest IP
      idx2 = 0;
      strncpy(tmp, result, BUF_SZ); // copy up to the size of the target buffer
      result2 = strtok_r( tmp , ipdelim, &svptr2 );
      while( result2 != NULL ) {
	ip.octet[idx2] = atoi(result2);
	result2 = strtok_r(NULL, ipdelim, &svptr2 );
	idx2++;
      }
      myind->chrome[G_DEST_IP] = ip.full;
      if (getUnique) 
	updateL(uniqueL[G_DEST_IP], uniqueC[G_DEST_IP], myind->chrome[G_DEST_IP]);
      break; // End of Dest IP

    case F_ATTACK: // attack name
      // FIXME : Is -1 good for this result?
      myind->chrome[G_ATTACK] = NONE;
      for ( i = 1; i < END_A; i++) {
	if ( strncmp(attacks[i], result, strlen( attacks[i] ) ) == 0 ) {
	  matched = 1;
	  myind->chrome[G_ATTACK] = i;
	  if (getUnique) 
	    updateL(uniqueL[G_ATTACK], uniqueC[G_ATTACK], myind->chrome[G_ATTACK]);
	}
      }
      break; // end of attack

    default:
      // Do Nothing
      ;
    }      
      //g_printf("|");
    idx++;
    result = strtok_r( NULL, delims, &svptr1 );
  }
    //g_printf("\n");
  
}


int load_audit(GSList **auditList, char *myfile )
{

  return load_audit_unique(auditList, myfile, NULL, NULL);
}


gint load_audit_unique(GSList **auditList, char *myfile, GHashTable *uniqueL[NUM_HTABLES], GHashTable *uniqueC[NUM_HTABLES][SUBH])
{
  FILE *input; // input file
  char *rd_buf; // buffer for reading input
  int nchars; // number of characters read
  gint nRecords = 0;
  size_t cur_sz = BUF_SZ;
  individual *myInd;

  // Empty the list
  if (*auditList != NULL)
    {
      g_slist_free(*auditList);
      *auditList = NULL;
    }

  // Open the file
  //g_printf("file to open %s\n",myfile);

  input = fopen(myfile, "r");
  if (input == NULL)
    {
      perror("Failed to open file");
      exit(-1);
    }
  // buffer for reading input
  rd_buf = (char *) malloc( BUF_SZ * sizeof(char) );

  nchars = getline(&rd_buf, &cur_sz, input);

  while (nchars != -1) {
    myInd = g_slice_new0(individual);
    global_individual_count++;

    if ( rd_buf[nchars-1] == '\n')
      rd_buf[nchars-1] = '\0'; // get rid of newline. Probably won't work on PC though \n\r
    // string description
    g_snprintf(myInd->desc, DESC_SZ, "%s",rd_buf );
    // g_printf("Read this %s\n",rd_buf);

    make_toks(rd_buf, nchars, myInd, uniqueL, uniqueC);
    *auditList = g_slist_append(*auditList, myInd);
    nRecords++;
    nchars = getline(&rd_buf, &cur_sz, input);
  }

  free(rd_buf);


  if (fclose(input) != 0) {
    perror("Failed to close");
    exit(-1);
  }
  return nRecords;

}

void copyeachL(gpointer a, gpointer b, gpointer userdata) {
  GArray * myArray;
  myArray = (GArray *)userdata;
  g_array_append_val( myArray, *(guint *)a);
}

void copyeachC(gpointer a, gpointer b, gpointer userdata) {
  GArray * myArray;
  myArray = (GArray *)userdata;
  g_array_append_val( myArray, *(guchar *)a);
}


int build_audit_array(GSList **auditList, GArray *arrayL[NUM_HTABLES],  
		      GArray *arrayC[NUM_HTABLES][SUBH],
		      char *myfile) {
  
  gint nRecords = 0; // Number of records in audit data
  gint i,j; // loop variables
  gint neg1 = -1;
  //  GArray *tmpArray;
  GHashTable *myHTableL[NUM_HTABLES];
  GHashTable *myHTableC[NUM_HTABLES][SUBH];

  // Initialize each array and put wildcard at the beginning
  
  for (i=0; i< NUM_HTABLES;i++) {
    //tmpArray = g_array_new(FALSE, FALSE, sizeof(guint));
    arrayL[i] = g_array_new(FALSE, FALSE, sizeof(guint));
    g_array_append_val (arrayL[i], neg1 );
    // use memcpy here instead!
    //arrayL[i] = tmpArray;
  }
  
  for (i=0; i< NUM_HTABLES;i++) 
    for (j=0; j<SUBH; j++) {
      //tmpArray = g_array_new(FALSE, FALSE, sizeof(guchar));
      arrayC[i][j] = g_array_new(FALSE, FALSE, sizeof(guchar));
      g_array_append_val (arrayC[i][j], neg1 );
      // use memcpy here instead!
      //arrayC[i][j] = tmpArray;
    }

  for (i=0; i< NUM_HTABLES;i++) 
    myHTableL[i] = g_hash_table_new_full(g_int_hash,g_int_equal, 
					 (GDestroyNotify)destroyL_key,
					 (GDestroyNotify)destroyL_value);

  for (i=0; i< NUM_HTABLES;i++) 
    for (j=0; j<SUBH; j++)
      myHTableC[i][j] = g_hash_table_new_full(g_char_hash,g_char_equal, 
					      (GDestroyNotify)destroyC_key,
					      (GDestroyNotify)destroyL_value);

  nRecords = load_audit_unique(auditList, myfile, myHTableL, myHTableC);

  // Load the data into arrays

  // Duration 
  // -1, Hours, Min, Sec
  //  0    1     2    3
  for (i=0; i< SUBH; i++)
    g_hash_table_foreach(myHTableC[G_DURATION][i], copyeachC, 
			 arrayC[G_DURATION][i]);

  // Service
  g_hash_table_foreach(myHTableL[G_SERVICE], copyeachL, 
		       arrayL[G_SERVICE]);


  // Source Port
  g_hash_table_foreach(myHTableL[G_SOURCE_PORT], copyeachL, 
		       arrayL[G_SOURCE_PORT]);

  // Dest Port
  g_hash_table_foreach(myHTableL[G_DEST_PORT], copyeachL, 
		       arrayL[G_DEST_PORT]);

  // Source IP Address
  // xxx, xxx, xxx, xxx
  //  0    1    2    3
  for (i=0; i< SUBH; i++)
    g_hash_table_foreach(myHTableC[G_SRC_IP][i], copyeachC, 
			 arrayC[G_SRC_IP][i]);

  // Dest IP Address
  // xxx, xxx, xxx, xxx
  //  0    1    2    3
  for (i=0; i< SUBH; i++)
    g_hash_table_foreach(myHTableC[G_DEST_IP][i], copyeachC, 
			 arrayC[G_DEST_IP][i]);

  // Attack
  g_hash_table_foreach(myHTableL[G_ATTACK], copyeachL, 
		       arrayL[G_ATTACK]);


  // free Hash tables. Don't want to leak memory
  for (i=0; i< NUM_HTABLES;i++) 
    g_hash_table_destroy(myHTableL[i]);

  for (i=0; i< NUM_HTABLES;i++) 
    for (j=0; j<SUBH; j++)
      g_hash_table_destroy(myHTableC[i][j]);

  return nRecords;
}
